/* $Id: lock.c,v 1.1 1998/12/11 23:59:30 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "hpe1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * HPE1432_INPUT_CHANS)

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d (0x%x)\n", progname, #func, _s, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d (0x%x)\n", progname, #func, _s, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: lock.c,v 1.1 1998/12/11 23:59:30 ericb Exp $";
static const char *progname;

int
init(int nmod, const int *laddr, ViPSession session, int verbose)
{
    char    str[NMOD_MAX * 4 + 32];
    int     i;

    (void) sprintf(str, "VXI0::");
    for (i = 0; i < nmod; i++)
	(void) sprintf(str + strlen(str), "%d,", laddr[i]);
    (void) sprintf(str + strlen(str) - 1, "::INSTR");

    if (verbose)
	(void) printf("Using %s\n", str);

    CHECK(hpe1432_init(str, 0, 1, session));
    return 0;
}

int
run(ViSession session, int verbose)
{
    if (verbose)
	(void) printf("Setting lock mode\n");
    CHECK(hpe1432_setLockMode(session, 1));

    if (verbose)
	(void) printf("Sleeping 10\n");
    (void) sleep(10);

    if (verbose)
	(void) printf("Done\n");
    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-uvV] [-L laddr] [-N nmod]\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    ViSession session;
    char   *p;
    int     laddr[NMOD_MAX];
    int     c, i, nmod, verbose;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    laddr[0] = 8;
    nmod = 1;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "L:N:uvV")) != -1)
	switch (c)
	{
	case 'L':
	    laddr[0] = strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    if (init(nmod, laddr, &session, verbose) < 0)
	return EXIT_FAILURE;
    if (run(session, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
